//
// (c) 2023 wesolutions GmbH
// All rights reserved.
//

import QtQml   2.12
import QtQuick 2.12
import QtQuick.Layouts 1.12

import wesual.Create 1.0
import wesual.Ui     1.0

import wesual.Ui.Private 1.0

import wesual.data.Core 1.0

Loader {
    id : picker

    property Context context
    property string type

    property LinkedDocument document
    property bool readOnly : false

    readonly property DocumentTypeInfo __typeInfo : Creator.typeInfo(type)
    property Component __customPicker

    signal documentPicked(LinkedDocument document)

    function updateCustomPicker() {
        if (__typeInfo) {
            const picker = __typeInfo.typeProperty("Picker");
            if (picker) {
                const comp = Qt.createComponent(picker);
                if (comp.status === Component.Ready) {
                    __customPicker = comp;
                } else {
                    console.error(comp.errorString());
                    __customPicker = null;
                }
            } else {
                __customPicker = null;
            }
        } else {
            __customPicker = null;
        }
    }

    sourceComponent : __customPicker || defaultPickerComponent

    onItemChanged : {
        if (item) {
            item.document = Qt.binding(function() {
                if (picker.document && picker.document.hasType(picker.type)) {
                    return picker.document;
                }

                return null;
            });

            if ("readOnly" in item) {
                item.readOnly = Qt.binding(() => picker.readOnly);
            }

            item.documentPicked.connect(function(doc) {
                picker.documentPicked(doc);
            });
        }
    }

    on__TypeInfoChanged : Qt.callLater(updateCustomPicker)

    Component.onCompleted : updateCustomPicker()

    Component {
        id : defaultPickerComponent

        MouseArea {
            id : defaultPicker

            property LinkedDocument document

            property bool readOnly : false

            signal documentPicked(LinkedDocument document)

            implicitWidth : 100
            implicitHeight : 25
            hoverEnabled : true
            activeFocusOnTab : true
            enabled : !picker.readOnly

            onClicked : popup.show()

            Keys.onPressed : {
                switch (event.key) {
                case Qt.Key_Enter:
                case Qt.Key_Return:
                case Qt.Key_Space:
                    popup.show();
                    event.accepted = true;
                    break;
                default:
                    event.accepted = false;
                    break;
                }
            }

            UiDropDownBackground {
                anchors.fill : parent
            }

            UiText {
                id : label

                anchors.fill : parent
                anchors.rightMargin : 24

                color : {
                    if (!enabled) {
                        return UiColors.getColor(UiColors.MediumLightGrey27);
                    } else if (readOnly) {
                        return UiColors.getColor(UiColors.MediumGrey45);
                    } else if (highlight) {
                        UiColors.getColor(UiColors.White);
                    } else {
                        return UiColors.getColor(UiColors.DarkGrey80);
                    }
                }
                leftPadding : 6
                verticalAlignment : Qt.AlignVCenter
                text : {
                    if (document) {
                        return document.label;
                    }

                    return "";
                }
            }

            PersistentTypeListing {
                id : listing

                context : picker.context
                type : picker.type
            }

            Component {
                id : delegate_

                UiDelegate {
                    function pick() {
                        defaultPicker.documentPicked(modelData);
                        popup.hide();
                    }

                    active : modelData === defaultPicker.document
                    implicitHeight : 26
                    width : parent ? parent.width : 100

                    onClicked : pick()

                    UiBorder {
                        anchors.fill : parent
                        visible : parent.activeFocus
                        borderWidth : 1
                        style : UiBorder.Solid
                        color : UiColors.getColor(UiColors.DarkerSelectionGreen)
                    }

                    UiText {
                        anchors.fill : parent
                        verticalAlignment : Qt.AlignVCenter
                        leftPadding : 6
                        rightPadding : 6
                        text : modelData ? modelData.label : ""
                        highlight : parent.active || parent.pressed
                    }

                    Keys.onSpacePressed : pick()
                    Keys.onEnterPressed : pick()
                    Keys.onReturnPressed : pick()
                }
            }

            UiPopupAnchor {
                id : popup

                readonly property bool flipped : distanceTop > distanceBottom
                readonly property real availableHeight : {
                    if (flipped) {
                        return distanceTop - 40;
                    } else {
                        return distanceBottom - 40;
                    }
                }

                anchors {
                    verticalCenter : parent.verticalCenter
                }
                triggerItem : picker

                onAboutToHide : picker.forceActiveFocus()

                popup : FocusScope {
                    id : popupItem

                    implicitWidth : picker.width
                    implicitHeight : Math.min(
                                         popup.availableHeight,
                                         content.implicitHeight + 4)
                    opacity : 0
                    enabled : false

                    states : State {
                        name : "visible"
                        when : popup.popupVisible

                        PropertyChanges {
                            target : popupItem
                            opacity : 1
                            enabled : true
                        }
                    }

                    transitions : Transition {
                        NumberAnimation {
                            property : "opacity"
                        }
                    }

                    Rectangle {
                        anchors.fill : parent
                        border {
                            width : 1
                            color : UiColors.getColor(UiColors.ControlGrey20)
                        }
                    }
                    y : popup.flipped ? -(height + 11) : 11

                    Keys.onUpPressed : {
                        event.accepted = true;
                        list.forceActiveFocus();
                        list.currentIndex = list.count - 1;
                    }

                    Keys.onDownPressed : {
                        event.accepted = true;
                        list.forceActiveFocus();
                    }

                    Component.onCompleted : forceActiveFocus()

                    ColumnLayout {
                        id : content

                        anchors.fill : parent
                        anchors.margins : 2
                        spacing : 2

                        UiSearchBox {
                            id : searchBox

                            focus : true

                            Layout.fillWidth : true
                            Layout.margins : 4
                        }

                        DocumentSorter {
                            id : sortedDocuments

                            documents : listing.documents
                            delay : 10

                            StringSortCriterion {
                                property : "label"
                            }
                        }

                        DocumentFilter {
                            id : filteredDocuments

                            documents : sortedDocuments.result
                            delay : 10

                            StringFilterCriterion {
                                property : "label"
                                contains : searchBox.searchString
                            }
                        }

                        UiVerticalScrollView {
                            Layout.fillHeight : true
                            Layout.fillWidth : true
                            Layout.preferredHeight : list.contentHeight

                            ListView {
                                id : list

                                activeFocusOnTab : true
                                keyNavigationEnabled : true
                                keyNavigationWraps : true
                                highlightMoveDuration : 0
                                spacing : 0
                                model : {
                                    const items = [ null ];
                                    const docs = filteredDocuments.result;
                                    for (let i = 0; i < docs.length; ++i) {
                                        items.push(docs[i]);
                                    }

                                    return items;
                                }
                                delegate : delegate_

                                onActiveFocusChanged : {
                                    if (activeFocus) {
                                        currentIndex = 0;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}
